//------------------------------------------------------------------------------
// Filename: iRoboBumper.ino
//------------------------------------------------------------------------------
// Description:
//    This is the Arduino based software that runs on the Arduino Pro Mini 5V
// microcontroller board for the iRoboBumper project.  It handles all of the
// embedded software tasks such as, initialization (i.e. determining the "home"
// position), controlling the servo to maintain the desired pan rate, reading
// the 2 PING))) sensors and calculating the distance based on time of flight
// and temperature data, and outputting the information via the XBee module.
//------------------------------------------------------------------------------
#include <Servo.h>

#define NOP { __asm__ __volatile__ ("nop\n\t"); }
#define DELAY_HALF_uS  { NOP NOP NOP NOP NOP NOP NOP NOP }
#define DELAY_1_uS { DELAY_HALF_uS DELAY_HALF_uS }
// These macros allow for very small precise delays are they simply use a 
// set of NOP instructions.  Our 16 MHz microcontroller will cause each NOP
// to be 62.5 microseconds; thus, 8 will be 0.5us and 16 will be 1us.

#define  LED_PIN                      13    // The LED is connected to pin 13

#define  STATE_INITIALIZING            0    // Values of the systemState
#define  STATE_RUNNING                 1    // variable

#define  ATOD_SCALE_FACTOR         40000    //  (5,000 mV / 2^10) * 2^13
#define  ATOD_SHIFT                13       //  (2^13 = 8192)

#define  ZERO_DEG_C_VOLTAGE        2633
#define  TEMPERATURE_CONV_FACTOR   1214
#define  TEMPERATURE_SHIFT         14

//--------------------------------------------------------
// General Global Variable
//--------------------------------------------------------

unsigned char systemState = STATE_INITIALIZING;

unsigned int  numFullPans = 0;
// Counter variable used to track the number of full 360 degree pans that
// have been made since power on.

volatile unsigned char adcCount = 0;
// Counter variable used to determine which ADC input to sample next.

volatile int  adcTemperatureValue;
unsigned long temperatureVoltage = 0;
unsigned long temperatureDegC = 0;
// Variables used to read the LM87 temperature sensor and convert the
// analog voltage to a temperature in degrees C.

volatile int  adcProximityValue;
unsigned long proximityVoltage = 0;
// Variables used to read the TCRT1010 proximity sensor.

volatile int  adcBatteryValue;
unsigned long batteryVoltage = 0;
// Variables used to read the estimated input battery voltage.  The voltage
// read is approximately half of the actual battery voltage do to the hardware
// voltage divider.

//--------------------------------------------------------
// PING sensor interface constant and variables
//--------------------------------------------------------
#define PING_PINS              0b00001100  // PING sensor 1 and 2 connected to
                                           // pins 2 and 3 respectively.  These
                                           // map to PORTD pins 2 and 3.
#define PING1_INT                       0  // INT0 on digital pin 2
#define PING2_INT                       1  // INT1 on digital pin 3

#define PING_HOLDOFF_TIME             750  // Holdoff time of PING sensor
#define PING_MM_PER_US              10863  // Scaled cm/us 
#define PING_TEMP_OFFSET               20  // Scaled (cm/us) / deg C offset
#define PING_DISTANCE_SHIFT_FACTOR     16  // # of bits to shift the calculated
                                           //    PING distance

volatile unsigned char timer2Count = 0;
// Counter used to start a PING))) reading every other timer2 event.
                                          
volatile unsigned long pingReadingStartTime;
// Time the last PING))) reading was started.

volatile bool ping1ReturnRcvd = false;
volatile bool ping2ReturnRcvd = false;
// Flags indicating if the two PING))) sensor's returns have been received
// These are volatile as they are updated in the ISRs and used by
// the main loop.

volatile unsigned long ping1Distance;
volatile unsigned long ping2Distance;
// Variable used to hold the distance determined by both PING))) sensor's.
// These are volatile as they are updated in the ISRs and outputted by
// the main loop.

volatile unsigned long pingTimeToMillimeters = PING_MM_PER_US;
// Variable used to hold the conversion factor for converting from the
// duration of the PING))) sensor's pulse (in microseconds) to distance
// in millimeters.  This value is updated based on temperature.

const unsigned long maxTime = 4294967295;
// Constant value used to gracefully handle the roll over of the system
// time.  The micros() function returns a unsigned long (i.e. 32 bit value)
// which means it will rollover approximately every 71.58 minutes.

//--------------------------------------------------------
// Servo interface constant and variables
//--------------------------------------------------------
#define SERVO_PIN                       9   // Servo control signal is
                                            // connected to pin 9

#define DESIRED_FULL_PAN_TIME     1500000   // Desired pan rate in microseconds
#define SERVO_SCALE_FACTOR             16   // Shift factor to convert pan rate
                                            // difference in microseconds to 
                                            // the adjustment to make to the
                                            // servo's PWM
                                            
Servo          servo;
// The Arduino Servo software object used to control the servo's speed

unsigned int   servoMicroseconds = 1460
;
// Variable used to hold the current setting of the servo.

unsigned long  currentPanTime = DESIRED_FULL_PAN_TIME;
// Varialbe used to hold the time it took took for the most recent
// full 360 degree pan of the system.

float angleAtPingReadingStart = 0.0f;
// Variable use to hold the estimated angle off of the "home" position at
// the start of the PING))) sensors' reading.

//--------------------------------------------------------
// Alignment Hole Processing Constants and Variables
//--------------------------------------------------------
#define NUM_HOLES          3                // Number of Alignment Holes

bool holeDetected = true;
// Flag used to signal if the proximity sensor is currently over a hole.

volatile bool processAlignmentHoles = false;
// Flag used to signal that the time for all holes have been updated.

unsigned char nextHole = 0;
// Index of the next hole the proximity sensor will pass.  Hole 0 is the marks
// "home".  During initialization this variable will be updated to correctly
// represent "home" no matter where the proximity sensor was relative to the
// holes at power up.

unsigned long prevHoleTimes[NUM_HOLES];
unsigned long holeTimes[NUM_HOLES];
// System time, in microseconds, that the proximity sensor detected the
// alignment holes during the current and prev pass.  These times are used
// to determine the current pan rate, which is used to change the servo
// speed, and estimate the angle off of "home" at the start of each PING)))
// sensor reading.

//-----------------------------------------------------------------------
// Forward Function Declarations
//-----------------------------------------------------------------------
void InitiatePingReading();
void Ping1ISR();
void Ping2ISR();
void ProcessAlignmentHoles();
void AdjustServoSpeed();
void InitializeTimer2();
void InitializeADC();
unsigned long CalculatePingDistance();
unsigned long CalculateDuration(unsigned long startTime,
                                unsigned long stopTime);

//-----------------------------------------------------------------------------
// Function: setup()
//-----------------------------------------------------------------------------
// Description:
//    Standard Arduino function that runs each time the board is powered on or
// reset.
//-----------------------------------------------------------------------------
void setup()
{  
   // Initialize digital pin 13 as an output.  This is the digital pin 
   // connected to the LED on the Arduino Pro Mini board.
   pinMode(LED_PIN, OUTPUT);

   // Initialize the serial interface to the XBee Module.
   Serial.begin(115200);
  
   // Attach pin 9 to the servo object and set the speed of the servo to the
   // default setting.  The speed will be updated once the current pan rate 
   // has been determined.
   servo.attach(SERVO_PIN);
   servo.writeMicroseconds(servoMicroseconds);
  
   // Initialize the Analog to Digital Convertor of the microcontroller.
   InitializeADC();
  
   // Initialize Timer2 for use to initialize PING))) sensor readings at a
   // 50 Hz rate.
   InitializeTimer2();
}


//-----------------------------------------------------------------------------
// Function: loop()
//-----------------------------------------------------------------------------
// Description:
//    Standard Arduino function that runs, over and over again. continuously
// until the board is powered off or reset.  This function handles the lower
// priority (i.e. non-timing critical) tasks.  This includes processing the
// alignment holes once per pan to update the servo speed and outputting the
// serial data via the XBee serial to wireless module once the system has
// entered the RUNNING state.
//-----------------------------------------------------------------------------
void loop()
{
   //--------------------------------------------------------
   //               Alignment Hole Processing
   //--------------------------------------------------------
   // Determine if the times for all alignment holes have been updated.  If so,
   // process the times to find the home position and/or determine the current
   // pan rate.
   if (processAlignmentHoles)
   {
      // Set the processAlignmentHoles flag to false to prevent processing
      // again until the next full pass.
      processAlignmentHoles = false;
     
      ProcessAlignmentHoles();
      
      // If we are in the RUNNING state then output low frequency data.  This
      // is data that is outputted once per full 360 degree pan.
      if (systemState == STATE_RUNNING)
      {
         Serial.print("S\t");
         Serial.print(currentPanTime);
         Serial.print("\t");
         Serial.print(servoMicroseconds);
         Serial.print("\t");
         Serial.print(batteryVoltage);
         Serial.print("\t");
         Serial.println(temperatureDegC);      
      }
   }
  
   // Now determine if we have finished initialization.  Normal processing
   // of the PING sensor data, communication, etc. does not start until
   // after we have finished intialization.
   if (systemState == STATE_RUNNING)
   {
      //--------------------------------------------------------
      //                  PING Sensor Processing
      //--------------------------------------------------------
      if (ping1ReturnRcvd && ping2ReturnRcvd)
      {
         // Since we are outputting the data now, set the flags back to false.
         ping1ReturnRcvd = false;
         ping2ReturnRcvd = false;

         // Calculate the temperature in degrees C based on ADC value of the
         // voltage output of the LM87 temperature sensor.
         temperatureVoltage =
             (ATOD_SCALE_FACTOR * adcTemperatureValue) >> ATOD_SHIFT;
         temperatureDegC =
             ((ZERO_DEG_C_VOLTAGE - temperatureVoltage) *
                 TEMPERATURE_CONV_FACTOR) >> TEMPERATURE_SHIFT;
  
         // Updated the PING))) duration to millimeter conversion factor
         // based on the newly calculated temperature.
         pingTimeToMillimeters = PING_MM_PER_US +
                                     (temperatureDegC * PING_TEMP_OFFSET);
  
         // Calculate the battery voltage based on the ADC value
         batteryVoltage =
             2 * (ATOD_SCALE_FACTOR * adcBatteryValue) >> ATOD_SHIFT;

         Serial.print(angleAtPingReadingStart);
         Serial.print("\t");
         Serial.print(ping1Distance);
         Serial.print("\t");
         Serial.print(ping2Distance);
         Serial.print("\t");
         Serial.println(proximityVoltage);
      }
   }
}


//-----------------------------------------------------------------------------
// Function: InitializeADC()
//-----------------------------------------------------------------------------
// Description:
//    This function intializes the Analog to Digital Converter (ADC) of the
// Atmel microcontroller.  We do not use the Arduino analogRead() function
// as it blocks until the conversion is done which takes approximately
// 100 microseconds.  Instead, we configure the ADC to generate an interrupt
// upon the completion of the conversion (i.e. the ISR(ADC_vect) function will
// execute).
//-----------------------------------------------------------------------------
void InitializeADC()
{
   // Disable all digital inputs on port C's analog pins
   DIDR0 = 0x3F;            
   
   // Measuring on ADC2, use AVcc with ext. cap at AREF
   ADMUX = 0x42;
   
   // AD channels MUX on, free running mode
   ADCSRB = 0x00;
   
   // AD-converter on, Single Conv Mode, Interrupt Enabled, prescaler = 64
   ADCSRA = 0xCE;
   
   // Start the conversion by setting bit 6 (=ADSC) in ADCSRA
   bitWrite(ADCSRA, 6, 1);
   
   // set interrupt flag
   sei();
}


//-----------------------------------------------------------------------------
// Function: InitializeTimer2()
//-----------------------------------------------------------------------------
// Description:
//    This function intializes Timer2 of the Atmel microcontroller.  We setup
// Timer2 to generate an interupt at 100 Hz and initiate the start of a
// PING))) sensor reading every other interrupt.  Thus, this is the mechanism
// to have PING))) sensor reading every 50 Hz (i.e. 20ms).  Upon the
// expiration the ISR(TIMER2_COMPA_vect) function will execute.
//-----------------------------------------------------------------------------
void InitializeTimer2()
{
   // Initialize the TCCR2A and TCCR2B registers to 0
   TCCR2A = 0;
   TCCR2B = 0;
   
   // Initialize counter value to 0
   TCNT2  = 0;
   
   // Set the Ooutput Compare Register A so that the timer event occurs
   // at a 100Hz rate (i.e. every 10ms).
   //   OCR2A = 16,000,000 / (100 * 1024) - 1   (must be <256)
   OCR2A = 155;
   
   // Turn on CTC mode
   TCCR2A = (1 << WGM21);
   
   // Set CS21 bit for 8 prescaler
   TCCR2B = (1 << CS22) | (1 << CS21) | (1 << CS20);
   
   // enable timer compare interrupt
   TIMSK2 |= (1 << OCIE2A);  
}


//-----------------------------------------------------------------------------
// Function: ISR(ADC_vect)
//-----------------------------------------------------------------------------
// Description:
//    This is the Interrupt Service Routine (ISR) that will execute whenever
// the ADC process completes.  This routine handles the basic processing
// (i.e. we want to minimize the amount of time spent in the ISR) and then
// initiates the next conversion.  We continuously perform 4 readings over and
// over.  The order of the readings are:
//     Proximity, Battery, Proximity, Temperature, repeat...
// This order was choosen to read have equally higher speed, spaced proximity
// sensor readings and lower frequency updates of the battery voltage and
// temperature.  Each sensor maps to the following ADC input:
//     ADC1 - Battery Voltage
//     ADC2 - Proximity Sensor
//     ADC3 - Temperature Sensor  
//-----------------------------------------------------------------------------
ISR(ADC_vect)
{
   int aval;
   
   aval = ADCL;        // store lower byte ADC
   aval += ADCH << 8;  // store higher bytes ADC

   // Determine which voltage was just sampled by the ADC.  The adcCount
   // value is used to cycle through the readings.
   if ((adcCount == 0) || (adcCount == 2))
   {
      // Save the proximity ADC and voltage values
      adcProximityValue = aval;
      proximityVoltage = (ATOD_SCALE_FACTOR * adcProximityValue) >> ATOD_SHIFT;

      // Determine if the proximity sensor was over a hole
      // on the previous pass.      
      if (holeDetected)
      {
         // If we were over a hole and the proximity voltage has risen above
         // the threshold voltage, change the flag to indicate we are no
         // longer over a hole.
         if (proximityVoltage > 1300)
         {
            digitalWrite(LED_PIN, LOW);
            holeDetected = false;  
         }
      }
      else
      {
         // If we are not over a hole, then determine if the proximity
         // voltage has fallen below the threshold voltage, change the flag
         // to indicate we are over a hole and update the timing
         // information, nextHole value, and set the processAlignmentHoles
         // flag if requied.
         if (proximityVoltage < 1100)
         {  
            holeDetected = true;
            
            // Store off the previous time this hole was detected and
            // then save the current time.
            prevHoleTimes[nextHole] = holeTimes[nextHole];
            holeTimes[nextHole] = micros();
            
            // Increment the nextPos index, wrapping around at NUM_HOLES.
            nextHole++;
            if (nextHole >= NUM_HOLES)
               nextHole = 0;

            // If we just processed hole 0 (i.e. the home position), set
            // the flag to process the alignment holes.
            if (nextHole == 1)
            {
               digitalWrite(LED_PIN, HIGH);
               processAlignmentHoles = true;            
            }
         }
      }

      // Set the AtoD Mux to ADC1 or ADC3 to sample the battery or
      // temperature sensor depending on the adcCount value.
      if (adcCount == 0)
         ADMUX = 0x41;
      else
         ADMUX = 0x43;
   }
   else if (adcCount == 1)
   {
      // Save the battery voltage reading.
      adcBatteryValue = aval;
     
      // Set the AtoD Mux to ADC2 to read the proximity sensor.
      ADMUX = 0x42;
   }
   else
   {
      // Save the temperature sensor reading.
      adcTemperatureValue = aval;
     
      // Set the AtoD Mux to ADC2 to read the proximity sensor.
      ADMUX = 0x42;
   }
   
   // Increment the adcCount variable from 0, 1, 2, 3, 0, 1, 2, 3...
   adcCount = (adcCount+1) & 0x3;

   // Start the next conversion.  We are operating in single converstion
   // mode, generating interrupts on completion.
   ADCSRA = 0xCE;
}


//-----------------------------------------------------------------------------
// Function: ISR(TIMER2_COMPA_vect)
//-----------------------------------------------------------------------------
// Description:
//    This is the Interrupt Service Routine (ISR) that will execute whenever
// Timer2 expires.  Timer2 has been configured to expire at a 100 Hz rate, but
// we only want to initiate a PING))) sensor reading at a 50 Hz rate.  Thus,
// the timer2Count variable is used to initiate a reading every other time
// this ISR runs.
//-----------------------------------------------------------------------------
ISR(TIMER2_COMPA_vect)
{
   unsigned long  timeSinceAtHome;
   
   // Increment the Timer2 count
   timer2Count++;
   
   // Determine if it is time to initiate another PING))) sensor command.  
   // We only start the PING))) command every other Timer2 event.
   if (timer2Count & 0x1)
   {
      // Record the current system time in microseconds.  This is used to
      // calculate the duration of the pulse from the PING))) sensor.
      pingReadingStartTime = micros();
      
      // Determine the angle off of the "home" position based on the time that
      // has occurred since we were at "home" (i.e. hole 0) and the duration
      // of the last full 360 degree pan.
      timeSinceAtHome = CalculateDuration(holeTimes[0], pingReadingStartTime);
      if (timeSinceAtHome > currentPanTime)
         timeSinceAtHome = currentPanTime;
      
      angleAtPingReadingStart = 360.0f * 
          ((float)timeSinceAtHome / (float)currentPanTime);

      // Initiate the PING))) sensor readings.     
      InitiatePingReading();
   }
}


//-----------------------------------------------------------------------------
// Function: InitiatePingReading()
//-----------------------------------------------------------------------------
// Description:
//    This routine handles initiate the PING))) sensors to take a reading;
// as well as, setting up / enabling the interrupts to occur when each sensor
// drives the return signal low.  The PING sensor is commanded to take a
// reading by driving a HIGH pulse of at least 2 microseconds (i.e. we drive
// them high for 3 microseconds).
//-----------------------------------------------------------------------------
void InitiatePingReading()
{
   // Configure PORTD pin 2 and 3 as outputs
   DDRD |= PING_PINS;       
  
   // Drive both PING))) sensor pins low for 2 microseconds.
   PORTD &= ~PING_PINS;     
   DELAY_1_uS;
   DELAY_1_uS;

   // Drive both PING))) sensor pins high for 3 microseconds
   PORTD |= PING_PINS;      
   DELAY_1_uS;
   DELAY_1_uS;
   DELAY_1_uS;

   // Drive both PING))) sensor pins low
   PORTD &= ~PING_PINS;

   // The PING sensor will now drive this signal high for a variable period of
   // time based on the distance to the object.  Thus, we change the mode of
   // the PING sensors' pins to INPUT.
   DDRD &= ~PING_PINS;
    
   // Set the return received flag for both sensors to false.
   ping1ReturnRcvd = false;
   ping2ReturnRcvd = false;
  
   // Clear the interrupts if they are currently set.  Writing a 1 to the
   // bottom 2 bits of the External Interrupt Flag register clears any
   // pending interrupts.
   EIFR = 0x3;
     
   // Finally attach the ISRs for each of the PING sensor pins.  The system
   // time will be read in the ISR so that the distance to the object can be
   // calculated for each PING sensor.
   attachInterrupt(PING1_INT, Ping1ISR, FALLING);
   attachInterrupt(PING2_INT, Ping2ISR, FALLING);
}



//-----------------------------------------------------------------------------
// Function: Ping1ISR()
//-----------------------------------------------------------------------------
// Description:
//    This is the interrupt service routine that will run when PING)))
// sensor 1 drives the signal line low.  The duration from the start of the
// reading until this interrupt is used to determine the distance to the
// object using time of flight.  This routine also set the flag to inform the
// loop() function that the reading has completed and detaches the interrupt.
//-----------------------------------------------------------------------------
void Ping1ISR(void)
{ 
  // Calculate the distance based on the time between the commanding of
  // the PING sensors to fire and the current time.
  ping1Distance = CalculatePingDistance();
  
  // Set the flag indicating that the return was received.
  ping1ReturnRcvd = true;
  
  // Detach (i.e. disable) the interrupt.  It will be reenabled at the start of
  // the next reading.
  detachInterrupt(PING1_INT);
}


//-----------------------------------------------------------------------------
// Function: Ping2ISR()
//-----------------------------------------------------------------------------
// Description:
//    This is the interrupt service routine that will run when PING)))
// sensor 2 drives the signal line low.  The duration from the start of the
// reading until this interrupt is used to determine the distance to the
// object using time of flight.  This routine also set the flag to inform the
// loop() function that the reading has completed and detaches the interrupt.
//-----------------------------------------------------------------------------
void Ping2ISR(void)
{
  // Calculate the distance based on the time between the commanding of
  // the PING sensors to fire and the current time.
  ping2Distance = CalculatePingDistance();

  // Set the flag indicating that the return was received.
  ping2ReturnRcvd = true;
  
  // Detach (i.e. disable) the interrupt.  It will be reenabled at the start of
  // the next reading.
  detachInterrupt(PING2_INT);  
}


//-----------------------------------------------------------------------------
// Function: CalculateDuration()
//-----------------------------------------------------------------------------
// Description:
//    This function calculates the difference between to system timestamp
// accounting for rollover if required.  This Arduino sketch uses the micros()
// function for all timing so the returned values is the the number of
// microseconds that elapsed between the startTime and stopTime.
//-----------------------------------------------------------------------------
unsigned long CalculateDuration(unsigned long startTime,
                                unsigned long stopTime)
{
   unsigned long time;
   
   if (stopTime > startTime)
   {
      time = stopTime - startTime;
   }
   else
   {
      time =  (maxTime - startTime) + stopTime;
   }
   
   return time;
}


//-----------------------------------------------------------------------------
// Function: CalculateDuration()
//-----------------------------------------------------------------------------
// Description:
//    This function calculates and returns the distance based on the amount
// of time between when the last PING))) sensor reading was started and the
// call to this function.  This routine is called from within the ISRs for
// both PING))) sensors.
//-----------------------------------------------------------------------------
unsigned long CalculatePingDistance()
{
   unsigned long currentTime, pingTime;
   unsigned long distance;
   
   // Get the current system time in microseconds.
   currentTime = micros();
   
   // Calculate the total time of the PING))) reading.
   pingTime = CalculateDuration(pingReadingStartTime, currentTime);
   
   // Subtract off the PING))) holdoff time.
   pingTime = pingTime - PING_HOLDOFF_TIME;
   
   // Convert the duration of the high pulse from the PING))) sensor
   // to millimeters using the pingTimeToMillimeters and shift factor.
   distance = pingTime * pingTimeToMillimeters;
   distance = distance >> PING_DISTANCE_SHIFT_FACTOR;
   
   return distance; 
}


//-----------------------------------------------------------------------------
// Function: ProcessAlignmentHoles()
//-----------------------------------------------------------------------------
// Description:
//    This function is called from the main loop() function whenever the
// proximity sensor processing within the ADC interrupt service routine
// indicates we have just passed the "home" hole.  On the second pass, we
// adjust the nextHole variable so that we consistently call the same hole
// "home" run to run no matter where the proximity sensor starts.  Starting
// on the fourth pass, we set the start to RUNNING and adjust the servo speed.
//-----------------------------------------------------------------------------
void ProcessAlignmentHoles()
{
   unsigned long temp;
   unsigned long hole0to1Time;
   unsigned long hole1to2Time;  
   unsigned long hole2to0Time;

   // Determine if we just completed the second full pan.  If so, we have
   // all of the information we need to determine where we are relative to
   // the "home" alignment hole.   
   if (numFullPans == 1)
   {
      // Calculate the times between each of the alignment holes over the
      // last pass.  NOTE:  The longest gap between alignment holes is
      // is immediately following the "home" hole.  This is where there is
      // approximately 180 degrees without a hole.
      hole0to1Time = CalculateDuration(prevHoleTimes[0], holeTimes[1]);
      hole1to2Time = CalculateDuration(holeTimes[1], holeTimes[2]);
      hole2to0Time = CalculateDuration(holeTimes[2], holeTimes[0]);
      
      // Determine if the time between hole 0 and 1 is the greatest,
      // If so, we are already sychronized with the "home" position.
      // Otherwise, we need to adjust so the nextHole so we are
      // synchronized.
      if ((hole0to1Time <= hole1to2Time) || 
          (hole0to1Time <= hole2to0Time))
      {
         // If the time between holes 1 and 2 is the greatest, then the
         // next hole we should detect is the "home" hole.  Otherwise,
         // the time between holes 2 and 0 must have been the greatest
         // and the next hole we detect should be hole 2. 
         if (hole1to2Time > hole2to0Time)
            nextHole = 0;
         else
            nextHole = 2;
      }
   }

   if (numFullPans > 2)
   {
      // Since, have made more than 2 passes over the alignment holes since
      // identifying the "home" position, we are ready to start processing
      // PING sensor date and communicating.  Thus, set the systemState to
      // RUNNING.
      systemState = STATE_RUNNING;
      
      // The AdjustServoSpeed function is called once per full pan to 
      // update the servo's PWM signal to maintain the desired
      // pan rate.
      AdjustServoSpeed();
   }
   
   // Increment the number of full pans
   numFullPans++;
}


//-----------------------------------------------------------------------------
// Function: AdjustServoSpeed()
//-----------------------------------------------------------------------------
// Description:
//    This function is called once per full pan after we are in the RUNNING
// state.  It calculates the difference between the last full 360 degree pan
// time and the desired pan rate.  It then updates the servo's PWM control
// based on the difference to keep the pan rate as close to the desired pan
// rate as possible.
//-----------------------------------------------------------------------------
void AdjustServoSpeed()
{
   long diffTime;

   // Calculate the time it took the system to pan a full 360 degrees.
   // Currently, we just use the time between the previous and current
   // detection of hole 0 (i.e. "home"). 
   currentPanTime = CalculateDuration(prevHoleTimes[0], holeTimes[0]);
   diffTime = DESIRED_FULL_PAN_TIME - currentPanTime;   
   servoMicroseconds += (diffTime >> SERVO_SCALE_FACTOR);

   // Update the servo's PWM control with the new value.   
   servo.writeMicroseconds(servoMicroseconds);   
}
